$(document).ready(function() {
	function showConfirm(message) {
		return new Promise((resolve) => {
			$("#modalMessage").text(message);
			$("#modalBackdrop").removeClass("hidden");
			$("#modalOk").off("click").on("click", function() {
				$("#modalBackdrop").addClass("hidden");
				resolve(true);
			});
			$("#modalCancel").off("click").on("click", function() {
				$("#modalBackdrop").addClass("hidden");
				resolve(false);
			});
		});
	}

    function showPrompt(message, defaultValue = "") {
        return new Promise((resolve) => {
            $("#promptMessage").text(message);
            $("#promptInput").val(defaultValue);
            $("#promptBackdrop").removeClass("hidden");
            $("#promptOk").off("click").on("click", function() {
                $("#promptBackdrop").addClass("hidden");
                resolve($("#promptInput").val().trim());
            });
            $("#promptCancel").off("click").on("click", function() {
                $("#promptBackdrop").addClass("hidden");
                resolve(null);
            });
        });
    }

	function showToast(message, duration = 2000) {
		const toast = $("#toast");
		toast.text(message).removeClass("hidden");
		setTimeout(() => toast.addClass("hidden"), duration);
	}

	function logout() {
		$.post("/logout", function(data) {
			location.reload();
		}).fail(function(xhr, status, error) {
			showToast("status: " + status);
		});
	}

	let currentPath = "/";

	function normalizeHref(href) {
		try {
			return new URL(href, window.location.origin).pathname;
		} catch (e) {
			return href;
		}
	}

	function listDir(path) {
		$.get("/list", {
			path: path
		}, function(data) {
            $("#fileTree").empty();
            let node;
            let responses = [];
            let parser = new DOMParser();
            let xmlDoc = parser.parseFromString(data.raw, "application/xml");
            let iterator = xmlDoc.evaluate("//*[local-name()='response']", xmlDoc, null, XPathResult.ANY_TYPE, null);
            while ((node = iterator.iterateNext())) {
                responses.push(node);
            }
			if (path !== "/") {
				let parentPath = "/" + path.split("/").filter(Boolean).slice(0, -1).join("/") +
					(path.split("/").filter(Boolean).length > 1 ? "/" : "");
				if (parentPath === "//") parentPath = "/";
				let li = $("<li>")
					.text("..")
					.addClass("flex justify-between items-center py-2 px-3 rounded hover:bg-gray-50 transition cursor-pointer")
					.addClass("folder")
					.data("href", parentPath)
					.click(function() {
						currentPath = parentPath;
						$("#currentPath").text(decodeURIComponent(currentPath));
						listDir(currentPath);
					});
				$("#fileTree").append(li);
			}
			let items = [];
			responses.forEach(resp => {
				let hrefEl = Array.from(resp.getElementsByTagName("*"))
					.find(el => el.localName === "href");
				if (!hrefEl) return;
				let href = normalizeHref(hrefEl.textContent.trim());
				if (href === path || href === path + "/") return;
				let displayName = decodeURIComponent(href.split("/").filter(Boolean).pop());
				let resType = Array.from(resp.getElementsByTagName("*"))
					.find(el => el.localName === "resourcetype");
				let isCollection = false;
				if (resType) {
					let coll = Array.from(resType.getElementsByTagName("*"))
						.find(el => el.localName === "collection");
					if (coll) isCollection = true;
				}
				items.push({
					href,
					displayName,
					isCollection
				});
			});
			items.sort((a, b) => {
				if (a.isCollection !== b.isCollection) {
					return b.isCollection - a.isCollection;
				}
				return a.displayName.localeCompare(b.displayName, undefined, {
					sensitivity: "base"
				});
			});
			items.forEach(item => {
				let li = $("<li>")
					.addClass("flex justify-between items-center py-2 px-3 rounded hover:bg-gray-50 transition cursor-pointer")
					.data("href", item.href);
				let nameSpan = $("<span>").text(item.displayName)
					.addClass(item.isCollection ? "font-semibold text-gray-800" : "text-gray-700 break-all");
				li.append(nameSpan);
				if (item.isCollection) {
					li.addClass("folder");
					li.click(function(e) {
						e.stopPropagation();
						currentPath = item.href.endsWith("/") ? item.href : item.href + "/";
						$("#currentPath").text(decodeURIComponent(currentPath));
						listDir(currentPath);
					});
				}
				let btnContainer = $("<div>").addClass("flex gap-2 ml-2");
				if (!item.isCollection) {
					let downloadBtn = $("<button>")
						.text("⬇️")
						.addClass("text-blue-500 hover:text-blue-700 transition")
						.click(e => {
							e.stopPropagation();
							const a = document.createElement("a");
							a.href = "/download?path=" + encodeURIComponent(item.href);
							a.download = item.displayName;
							document.body.appendChild(a);
							a.click();
							document.body.removeChild(a);
						});
					btnContainer.append(downloadBtn);
				}
                let renameBtn = $("<button>")
                    .text("✏️")
                    .addClass("text-green-500 hover:text-green-700 transition")
                    .click(async function(e) {
                        e.stopPropagation();
                        let newName = await showPrompt("Enter new name:", item.displayName);
                        if (newName && newName !== item.displayName) {
                            let src = item.href;
                            let parts = src.split("/");
                            let lastPart = parts.pop();
                            let decodedParts = parts.map(p => decodeURIComponent(p));
                            let dest = decodedParts.join("/") + "/" + newName;
                            if (item.isCollection && !dest.endsWith("/")) dest += "/";
                            $.post("/move", {
                                src_path: src,
                                dest_path: dest
                            }, function(data) {
                                if (data.status === "error") {
                                    showToast("Rename failed: " + (data.message || "Already exists"));
                                } else {
                                    showToast("status: " + data.status);
                                    listDir(currentPath);
                                }
                            });
                        }
                    });
                btnContainer.append(renameBtn);
				let deleteBtn = $("<button>")
					.text("🗑️ ")
					.addClass("text-red-500 hover:text-red-700 transition")
					.click(async function(e) {
						e.stopPropagation();
						const confirmed = await showConfirm("Delete " + item.displayName + "?");
						if (confirmed) {
							$.post("/delete", {
								path: item.href
							}, function(data) {
								showToast("status: " + data.status);
								listDir(currentPath);
							});
						}
					});
				btnContainer.append(deleteBtn);
				li.append(btnContainer);
				$("#fileTree").append(li);
			});

			makeDraggable();
		});
	}

	function makeDraggable() {
		$("#fileTree li").off("dragstart dragover dragleave drop");
		$("#fileTree li").attr("draggable", true);
		$("#fileTree li").on("dragstart", function(e) {
			e.originalEvent.dataTransfer.setData("text/plain", $(this).data("href"));
		});
		$("#fileTree li.folder").on("dragover", function(e) {
			e.preventDefault();
			$(this).css("background-color", "#ddf");
		});
		$("#fileTree li.folder").on("dragleave", function(e) {
			$(this).css("background-color", "");
		});
		$("#fileTree li.folder").on("drop", function(e) {
			e.preventDefault();
			$(this).css("background-color", "");
			let src = e.originalEvent.dataTransfer.getData("text/plain");
			let folderHref = $(this).data("href");
			src = src.replace(/^https?:\/\/[^/]+/, "");
			folderHref = folderHref.replace(/^https?:\/\/[^/]+/, "");
            let folderHrefParts = folderHref.split("/")
            let decodedFolderHrefParts = folderHrefParts.map(p => decodeURIComponent(p));
            let decodedFolderHref = decodedFolderHrefParts.join("/")
			if (!decodedFolderHref.endsWith("/")) decodedFolderHref += "/";
			let name = src.split("/").filter(Boolean).pop();
            let decodeName = decodeURIComponent(name);
            let dest = decodedFolderHref + decodeName;
			$.post("/move", {
				src_path: src,
				dest_path: dest
			}, function(data) {
				showToast("status: " + data.status);
				listDir(currentPath);
			});
		});
	}

	function uploadFiles(files, callback) {
		let uploadedCount = 0;
		for (let i = 0; i < files.length; i++) {
			let file = files[i];
			let formData = new FormData();
			formData.append("file", file);
			formData.append("filename", file.name);
			formData.append("path", currentPath);
			$.ajax({
				url: "/upload",
				type: "POST",
				data: formData,
				processData: false,
				contentType: false,
				success: function(data) {
					uploadedCount++;
					if (uploadedCount === files.length) {
						listDir(currentPath);
						if (callback) callback();
					}
				}
			});
		}
	}

	$("#logoutBtn").click(function(e) {
		e.preventDefault();
		logout();
	});

	$("#uploadBtn").click(function(e) {
		e.preventDefault();
		let files = $("#uploadFile")[0].files;
		if (files.length === 0) return;
		uploadFiles(files, function() {
			$("#uploadFile").val("");
		});
	});

	$("#fileContainer").on("dragover", function(e) {
		e.preventDefault();
		$(this).css("background-color", "#eef");
	});

	$("#fileContainer").on("dragleave", function(e) {
		e.preventDefault();
		$(this).css("background-color", "#fff");
	});

	$("#fileContainer").on("drop", function(e) {
		e.preventDefault();
		$(this).css("background-color", "#fff");
		let files = e.originalEvent.dataTransfer.files;
		uploadFiles(files);
	});

	$("#mkdirBtn").click(function() {
		let folder = $("#mkdirName").val();
		$.post("/mkdir", {
			path: currentPath + folder + "/"
		}, function(data) {
			showToast("status: " + data.status);
			listDir(currentPath);
			$("#mkdirName").val("");
		});
	});

	listDir(currentPath);
});
