import os
import io
import requests
from urllib.parse import quote
from requests.auth import HTTPBasicAuth
from pydantic import BaseModel
from starlette.middleware.sessions import SessionMiddleware
from fastapi import FastAPI, Request, Form, Depends, UploadFile, File
from fastapi.responses import HTMLResponse, JSONResponse, RedirectResponse, StreamingResponse
from fastapi.staticfiles import StaticFiles
from fastapi.middleware.cors import CORSMiddleware

app = FastAPI()
app.add_middleware(SessionMiddleware, secret_key="super-secret-key")
app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"],
    allow_methods=["*"],
    allow_headers=["*"],
)

WEBDAV_HOST = os.getenv("WEBDAV_HOST", "http://webdav:8888")
WEBDAV_USER = os.getenv("WEBDAV_USER", "admin")
WEBDAV_PASS = os.getenv("WEBDAV_PASS", "pass")

app.mount("/static", StaticFiles(directory="./static"), name="static")

# ---------- Models ----------
class LoginRequest(BaseModel):
    username: str
    password: str

# ---------- Auth ----------
def get_current_user(request: Request):
    user = request.session.get("user")
    if not user:
        return None
    return user

def require_login(user = Depends(get_current_user)):
    if not user:
        return RedirectResponse(url="/login")
    return user

# ---------- Routes ----------
@app.get("/", response_class=HTMLResponse)
async def root(request: Request):
    user = request.session.get("user")
    if user:
        with open("./templates/index.html", "r", encoding="utf-8") as f:
            return HTMLResponse(f.read())
    else:
        with open("./templates/login.html", "r", encoding="utf-8") as f:
            return HTMLResponse(f.read())

@app.post("/login")
async def login(req: LoginRequest, request: Request):
    if req.username == WEBDAV_USER and req.password == WEBDAV_PASS:
        request.session["user"] = req.username
        return JSONResponse({"success": True})
    return JSONResponse({"success": False})

@app.post("/logout")
async def logout(request: Request):
    request.session.clear()
    return JSONResponse({"success": True})

# ---------- WebDAV Proxy ----------
def strip_host(path: str):
    return path.replace(WEBDAV_HOST, "")

@app.get("/list")
async def list_dir(path: str, username = Depends(require_login)):
    url = f"{WEBDAV_HOST}{path}"
    headers = {"Depth": "1"}
    resp = requests.request("PROPFIND", url, headers=headers, auth=HTTPBasicAuth(WEBDAV_USER, WEBDAV_PASS))
    return JSONResponse({"raw": resp.text})

@app.post("/upload")
async def upload_file(path: str = Form(...), file: UploadFile = File(...), username = Depends(require_login)):
    url = f"{WEBDAV_HOST}{path}/{file.filename}"
    data = await file.read()
    resp = requests.put(url, data=data, auth=HTTPBasicAuth(WEBDAV_USER, WEBDAV_PASS))
    return JSONResponse({"status": resp.status_code})

@app.post("/mkdir")
async def mkdir(path: str = Form(...), username = Depends(require_login)):
    url = f"{WEBDAV_HOST}{path}"
    resp = requests.request("MKCOL", url, auth=HTTPBasicAuth(WEBDAV_USER, WEBDAV_PASS))
    return JSONResponse({"status": resp.status_code})

@app.post("/delete")
async def delete(path: str = Form(...), username = Depends(require_login)):
    url = f"{WEBDAV_HOST}{path}"
    resp = requests.delete(url, auth=HTTPBasicAuth(WEBDAV_USER, WEBDAV_PASS))
    return JSONResponse({"status": resp.status_code})

@app.post("/move")
async def move_item(src_path: str = Form(...), dest_path: str = Form(...), username = Depends(require_login)):
    src_path = strip_host(src_path)
    dest_path = strip_host(dest_path)
    encoded_dest = quote(dest_path, safe="/")
    url = f"{WEBDAV_HOST}{src_path}"
    headers = {"Destination": f"{encoded_dest}"}
    resp = requests.request("MOVE", url, headers=headers, auth=HTTPBasicAuth(WEBDAV_USER, WEBDAV_PASS))
    return JSONResponse({"status": resp.status_code})

@app.get("/download")
async def download(path: str, username = Depends(require_login)):
    url = f"{WEBDAV_HOST}{path}"
    resp = requests.get(url, auth=HTTPBasicAuth(WEBDAV_USER, WEBDAV_PASS), stream=True)
    filename = path.split("/")[-1]
    return StreamingResponse(
        io.BytesIO(resp.content),
        media_type="application/octet-stream",
        headers={"Content-Disposition": f"attachment; filename*=UTF-8''{filename}"}
    )

