import os
import secrets
from fastapi import FastAPI, Request, Depends, HTTPException, status
from fastapi.security import HTTPBasic, HTTPBasicCredentials
from handler import *
from database import init_db

app = FastAPI()
security = HTTPBasic()

WEBDAV_USER = os.getenv("WEBDAV_USER", "admin")
WEBDAV_PASS = os.getenv("WEBDAV_PASS", "password")

init_db()

def auth(credentials: HTTPBasicCredentials = Depends(security)):
    correct_user = secrets.compare_digest(credentials.username, WEBDAV_USER)
    correct_pass = secrets.compare_digest(credentials.password, WEBDAV_PASS)
    if not (correct_user and correct_pass):
        raise HTTPException(status_code=status.HTTP_401_UNAUTHORIZED, detail="Unauthorized")
    return credentials.username

@app.options("/{full_path:path}")
async def options(request: Request, full_path: str, username: str = Depends(auth)):
    return await handle_options(request)

@app.put("/{full_path:path}")
async def put(request: Request, full_path: str, username: str = Depends(auth)):
    return await handle_put(request, full_path)

@app.get("/{full_path:path}")
async def get(request: Request, full_path: str, username: str = Depends(auth)):
    return await handle_get(request, full_path)

@app.delete("/{full_path:path}")
async def delete(request: Request, full_path: str, username: str = Depends(auth)):
    return await handle_delete(request, full_path)

@app.api_route("/{full_path:path}", methods=["MOVE"])
async def move(request: Request, full_path: str):
    return await handle_move(request, full_path)

@app.api_route("/{full_path:path}", methods=["PROPFIND"])
async def propfind(request: Request, full_path: str):
    return await handle_propfind(request, full_path)

@app.api_route("/{full_path:path}", methods=["MKCOL"])
async def mkcol(request: Request, full_path: str, username: str = Depends(auth)):
    return await handle_mkcol(request, full_path)

@app.post("/{full_path:path}/lock")
async def lock(request: Request, full_path: str, username: str = Depends(auth)):
    return await handle_lock(request, full_path)

@app.post("/{full_path:path}/unlock")
async def unlock(request: Request, full_path: str, username: str = Depends(auth)):
    return await handle_unlock(request, full_path)

