import os
import uuid
import shutil
import mimetypes
import xml.etree.ElementTree as ET
from pathlib import Path
from urllib.parse import unquote
from fastapi import Request, HTTPException
from fastapi.responses import  Response, StreamingResponse
from database import SessionLocal, Lock
from datetime import datetime, timedelta

ROOT_DIR = "./data"
os.makedirs(ROOT_DIR, exist_ok=True)

CHUNK_SIZE = 1024 * 1024

# -----------------------------
# 안전한 경로 처리
# -----------------------------
def safe_path(full_path: str) -> str:
    root = Path(ROOT_DIR).resolve()
    target = (root / full_path.lstrip("/")).resolve()
    if not str(target).startswith(str(root)):
        raise HTTPException(status_code=403, detail="Access outside root directory forbidden")
    return str(target)

# -----------------------------
# OPTIONS
# -----------------------------
async def handle_options(request: Request):
    headers = {
        "Allow": "OPTIONS, GET, PUT, DELETE, PROPFIND, MKCOL, MOVE, LOCK, UNLOCK",
        "DAV": "1,2,extended-mkcol",
    }
    return Response(status_code=200, headers=headers)

# -----------------------------
# MKCOL
# -----------------------------
async def handle_mkcol(request: Request, full_path: str):
    path = safe_path(full_path)
    if os.path.exists(path):
        raise HTTPException(status_code=405, detail="Collection already exists")
    os.makedirs(path, exist_ok=True)
    return Response(status_code=201)

# -----------------------------
# GET
# -----------------------------
async def handle_get(request: Request, full_path: str):
    path = safe_path(full_path)
    if not os.path.exists(path):
        raise HTTPException(status_code=404)
    if os.path.isdir(path):
        raise HTTPException(status_code=400, detail="Cannot GET a directory")
    file_size = os.path.getsize(path)
    mime_type = mimetypes.guess_type(path)[0] or "application/octet-stream"
    range_header = request.headers.get("range")
    def iter_file(start: int, end: int):
        with open(path, "rb") as f:
            f.seek(start)
            remaining = end - start + 1
            while remaining > 0:
                chunk_size = min(CHUNK_SIZE, remaining)
                data = f.read(chunk_size)
                if not data:
                    break
                remaining -= len(data)
                yield data
    if range_header:
        range_value = range_header.strip().lower().split("=")[-1]
        start_str, end_str = range_value.split("-")
        start = int(start_str) if start_str else 0
        end = int(end_str) if end_str else file_size - 1
        if start >= file_size:
            raise HTTPException(status_code=416, detail="Requested Range Not Satisfiable")
        headers = {
            "Content-Range": f"bytes {start}-{end}/{file_size}",
            "Accept-Ranges": "bytes",
            "Content-Length": str(end - start + 1),
        }
        return StreamingResponse(
            iter_file(start, end),
            status_code=206,
            headers=headers,
            media_type=mime_type,
        )
    headers = {
        "Accept-Ranges": "bytes",
        "Content-Length": str(file_size),
    }
    return StreamingResponse(
        iter_file(0, file_size - 1),
        media_type=mime_type,
        headers=headers,
    )

# -----------------------------
# PUT
# -----------------------------
async def handle_put(request: Request, full_path: str):
    path = safe_path(full_path)
    os.makedirs(os.path.dirname(path), exist_ok=True)
    content = await request.body()
    with open(path, "wb") as f:
        f.write(content)
    return Response(status_code=201)

# -----------------------------
# DELETE (재귀 삭제)
# -----------------------------
async def handle_delete(request: Request, full_path: str):
    path = safe_path(full_path)
    if not os.path.exists(path):
        raise HTTPException(status_code=404)
    if os.path.isdir(path):
        shutil.rmtree(path)
    else:
        os.remove(path)
    return Response(status_code=204)

# -----------------------------
# MOVE
# -----------------------------
async def handle_move(request: Request, full_path: str):
    src_path = safe_path(full_path)
    if not os.path.exists(src_path):
        return Response(status_code=404)
    dest_header = request.headers.get("Destination")
    if not dest_header:
        return Response(status_code=400, content="Missing Destination header")
    dest_path = unquote(dest_header)
    if dest_path.startswith("/"):
        dest_path = dest_path[1:]
    dest_path = safe_path(dest_path)
    overwrite = request.headers.get("Overwrite", "T")
    if os.path.exists(dest_path):
        if overwrite.upper() != "T":
            return Response(status_code=412)
        if os.path.isdir(dest_path):
            shutil.rmtree(dest_path)
        else:
            os.remove(dest_path)
    try:
        os.makedirs(os.path.dirname(dest_path), exist_ok=True)
        shutil.move(src_path, dest_path)
    except Exception as e:
        return Response(status_code=500, content=str(e))
    return Response(status_code=201)

# -----------------------------
# LOCK
# -----------------------------
async def handle_lock(request: Request, full_path: str):
    session = SessionLocal()
    token = str(uuid.uuid4())
    timeout = datetime.utcnow() + timedelta(minutes=5)
    lock = Lock(path=full_path, owner="unknown", token=token, timeout=timeout)
    session.add(lock)
    session.commit()
    session.close()
    headers = {"Lock-Token": f"<{token}>"}
    return Response(status_code=200, headers=headers)

# -----------------------------
# UNLOCK
# -----------------------------
async def handle_unlock(request: Request, full_path: str):
    token = request.headers.get("Lock-Token")
    if not token:
        raise HTTPException(status_code=400, detail="Lock-Token header required")
    session = SessionLocal()
    lock = session.query(Lock).filter(Lock.path == full_path, Lock.token == token.strip("<>")).first()
    if not lock:
        raise HTTPException(status_code=409, detail="No lock found")
    session.delete(lock)
    session.commit()
    session.close()
    return Response(status_code=204)

# -----------------------------
# PROPFIND
# -----------------------------
async def handle_propfind(request: Request, full_path: str):
    path = safe_path(full_path)
    if not os.path.exists(path):
        raise HTTPException(status_code=404)
    depth = request.headers.get("Depth", "1")
    multistatus = ET.Element("d:multistatus", attrib={"xmlns:d": "DAV:"})
    def add_response(p: str):
        href = ET.SubElement(multistatus, "d:response")
        rel_path = os.path.relpath(p, ROOT_DIR).replace("\\", "/")
        if rel_path == ".":
            rel_path = ""
        ET.SubElement(href, "d:href").text = "/" + rel_path
        propstat = ET.SubElement(href, "d:propstat")
        prop = ET.SubElement(propstat, "d:prop")
        ET.SubElement(prop, "d:displayname").text = os.path.basename(p) or "/"
        res_type = ET.SubElement(prop, "d:resourcetype")
        if os.path.isdir(p):
            ET.SubElement(res_type, "d:collection")
        ET.SubElement(prop, "d:getcontentlength").text = str(os.path.getsize(p)) if os.path.isfile(p) else "0"
        ET.SubElement(prop, "d:getlastmodified").text = datetime.utcfromtimestamp(os.path.getmtime(p)).strftime("%a, %d %b %Y %H:%M:%S GMT")
        ET.SubElement(propstat, "d:status").text = "HTTP/1.1 200 OK"
    is_root = os.path.abspath(path) == os.path.abspath(ROOT_DIR)
    if is_root or depth == "0":
        add_response(path)
    if os.path.isdir(path):
        for child in os.listdir(path):
            child_path = os.path.join(path, child)
            add_response(child_path)
    xml_str = ET.tostring(multistatus, encoding="utf-8", method="xml")
    return Response(content=xml_str, media_type="application/xml", status_code=207)

